package graph;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Set;

/*
 * A graph object <tt>G<V, E></tt> contains a set <tt>V</tt> of vertices and a set <tt>E</tt>
 * of edges.  Each edge e=(v, u) in E connects v to u. 
 * The graph being implemented is undirected.
 * 
 * For more information
 * about graphs and their related definitions see 
 * <a href="http://mathworld.wolfram.com/Graph.html">
 * http://mathworld.wolfram.com/Graph.html</a>.
 * 
 * <p>Through generics, a graph can be typed to specific classes for vertices
 * <code>V</code> and edges <code>E</code>. Such a graph can contain
 * vertices of type <code>V</code> and Edges of type <code>
 * E</code>.</p>
 * 
 * @param <V> vertex type
 * @param <E> edge type
 */
public class Graph<V, E> {
	
	protected HashMap<V, Integer> v2id;
	protected HashMap<Integer, V> id2v;

	/*
	 * Hashmap containing all vertices as keys, and their respective adjacency lists
	 * as elements.
	 */
	protected HashMap<V, ArrayList<E>> adjMap;
	
	/*
	 * Edge container storing the endpoints of an edge.
	 */
	protected class EdgeContainer {
		V src, tar;
		public EdgeContainer(V src, V tar) {
			this.src = src;
			this.tar = tar;
		}
	}
	
	/*
	 * Hashmap storing all edges as keys and their respective endpoints as 
	 * elements.
	 */
	protected HashMap<E, EdgeContainer> edges;
	
	public Graph() {
		adjMap = new HashMap<V, ArrayList<E>>();
		edges = new HashMap<E, EdgeContainer>();
		v2id = new HashMap<V, Integer>();
		id2v = new HashMap<Integer, V>();
	}
	
	/**
     * Returns an edge connecting source vertex to target vertex if such
     * vertices and such edge exist in this graph. Otherwise returns <code>
     * null</code>. If any of the specified vertices is <code>null</code>
     * returns <code>null</code>
     *
     *
     * @param src source vertex of the edge.
     * @param tar target vertex of the edge.
     *
     * @return an edge connecting source vertex to target vertex.
     */
	public E getEdge(V src, V tar) {
		for (E e : edges.keySet()) {
			EdgeContainer ep = edges.get(e);
			if ((src.equals(ep.src) && tar.equals(ep.tar))
					|| (tar.equals(ep.src)&& src.equals(ep.tar) )) {
				return e;
			}
		}
		return null;
	}
	
	/**
     * Adds the specified edge to this graph, going from the source vertex to
     * the target vertex. More formally, adds the specified edge, <code>
     * e</code>, to this graph if this graph contains no edge <code>e2</code>
     * such that <code>e2.equals(e)</code>. If this graph already contains such
     * an edge, the call leaves this graph unchanged and returns <tt>false</tt>.
     * If the edge was added to the graph, returns <code>
     * true</code>.
     *
     * <p>The source and target vertices must already be contained in this
     * graph. If they are not found in graph IllegalArgumentException is
     * thrown.</p>
     *
     * @param src source vertex of the edge.
     * @param tar target vertex of the edge.
     * @param data edge to be added to this graph.
     *
     * @return <tt>true</tt> if this graph did not already contain the specified
     * edge.
     *
     * @throws IllegalArgumentException if source or target vertices are not
     * found in the graph.
     * @throws NullPointerException if any of the specified vertices is <code>
     * null</code>.
     *
     */
    public boolean addEdge(E data, V src, V tar) {
    	if (src == null || tar == null) {
    		throw new NullPointerException();
    	}
    	
    	if (adjMap.get(tar) == null || adjMap.get(src) == null) {
    		throw new IllegalArgumentException();
    	}
    	
    	if (edges.containsKey(data)) {
    		return false;
    	}
    	
    	edges.put(data, new EdgeContainer(src, tar));
    	adjMap.get(src).add(data);
    	adjMap.get(tar).add(data);
    	return true;
    }
    
    /**
     * Adds the specified vertex to this graph if not already present. More
     * formally, adds the specified vertex, <code>v</code>, to this graph if
     * this graph contains no vertex <code>u</code> such that <code>
     * u.equals(v)</code>. If this graph already contains such vertex, the call
     * leaves this graph unchanged and returns <tt>false</tt>. In combination
     * with the restriction on constructors, this ensures that graphs never
     * contain duplicate vertices.
     *
     * @param v vertex to be added to this graph.
     *
     * @return <tt>true</tt> if this graph did not already contain the specified
     * vertex.
     *
     * @throws NullPointerException if the specified vertex is <code>
     * null</code>.
     */
    public boolean addVertex(V v) {
    	if (v == null) {
    		throw new NullPointerException();
    	}
    	
    	if (adjMap.containsKey(v)) {
    		return false;
    	}
    	
    	Integer vid = adjMap.size();
    	adjMap.put(v, new ArrayList<E>());
    	v2id.put(v,  vid);
    	id2v.put(vid, v);
    	return true;
    }
    
    /**
     * Get the vertex with the specified id.
     *
     * @param The vertex id to retrieve.
     *
     * @return The vertex with the associated id.
     *
     * @throws IllegalArgumentException no vertex has the corresponding id.
     */
    public V getVertexById(int id) {
    	if (!id2v.containsKey(id)) {
    		throw new IllegalArgumentException();
    	}
    	
    	return id2v.get(id);  	
    }
    
    /**
     * Get vertex id
     *
     * @param v vertex to get the id from.
     *
     * @return the id of v
     *
     * @throws IllegalArgumentException if v does not exist in the graph.
     * @throws NullPointerException vertex is null
     */
    public int getVertexId(V v) {
    	if (v == null) {
    		throw new NullPointerException();
    	}
    	
    	if (!v2id.containsKey(v)) {
    		throw new IllegalArgumentException();
    	}
    	
    	return v2id.get(v);  	
    }
    
    /**
     * Returns <tt>true</tt> if and only if this graph contains an edge going
     * from the source vertex to the target vertex. In undirected graphs the
     * same result is obtained when source and target are inverted. If any of
     * the specified vertices does not exist in the graph, or if is <code>
     * null</code>, returns <code>false</code>.
     *
     * @param src source vertex of the edge.
     * @param tar target vertex of the edge.
     *
     * @return <tt>true</tt> if this graph contains the specified edge.
     */
    public boolean containsEdge(V src, V tar) {
    	for (EdgeContainer ec : edges.values()) {
    		if (ec.src.equals(src) && ec.tar.equals(tar)
    				|| ec.src.equals(tar) && ec.tar.equals(src)) {
    			return true;
    		}
    	}
 
    	return false;
    }
    
    /**
     * Returns <tt>true</tt> if this graph contains the specified vertex. More
     * formally, returns <tt>true</tt> if and only if this graph contains a
     * vertex <code>u</code> such that <code>u.equals(v)</code>. If the
     * specified vertex is <code>null</code> returns <code>false</code>.
     *
     * @param v vertex whose presence in this graph is to be tested.
     *
     * @return <tt>true</tt> if this graph contains the specified vertex.
     */
    public boolean containsVertex(V v) {
    	return adjMap.containsKey(v);
    }
    
    /**
     * Returns a set of the edges contained in this graph. The set is backed by
     * the graph, so changes to the graph are reflected in the set. If the graph
     * is modified while an iteration over the set is in progress, the results
     * of the iteration are undefined.
     *
     *
     * @return a set of the edges contained in this graph.
     */
     public Set<E> edgeSet() {
    	 return edges.keySet();
     }
     
     /**
      * Returns a set of all edges touching the specified vertex. If no edges are
      * touching the specified vertex returns an empty set.
      *
      * @param vertex the vertex for which a set of touching edges is to be
      * returned.
      *
      * @return a set of all edges touching the specified vertex.
      *
      * @throws IllegalArgumentException if vertex is not found in the graph.
      * @throws NullPointerException if vertex is <code>null</code>.
      */
     public ArrayList<E> edgesOf(V v) {
    	 if (v == null) {
    		 throw new NullPointerException();
    	 }
    	 
    	 if (adjMap.get(v) == null) {
    		 throw new IllegalArgumentException();
    	 }
    	 return adjMap.get(v);
     }
      
     /**
      * Returns a set of the vertices contained in this graph. The set is backed
      * by the graph, so changes to the graph are reflected in the set. If the
      * graph is modified while an iteration over the set is in progress, the
      * results of the iteration are undefined.
      *
      *
      * @return a set view of the vertices contained in this graph.
      */
     public Set<V> vertexSet() {
    	 return adjMap.keySet();
     }
     
     /**
      * Returns the source vertex of an edge. For an undirected graph, source and
      * target are distinguishable designations (but without any mathematical
      * meaning).
      *
      * @param e edge of interest
      *
      * @return source vertex
      * 
      * @throws IllegalArgumentException if edge is not found in the graph.
      */
     public V getEdgeSource(E e) {
    	 if (!edges.containsKey(e)) {
    		 throw new IllegalArgumentException();
    	 }
    	 return edges.get(e).src;
     }

     /**
      * Returns the target vertex of an edge. For an undirected graph, source and
      * target are distinguishable designations (but without any mathematical
      * meaning).
      *
      * @param e edge of interest
      *
      * @return target vertex
      * 
      * @throws IllegalArgumentException if edge is not found in the graph.
      */
     public V getEdgeTarget(E e) {
    	 if (!edges.containsKey(e)) {
    		 throw new IllegalArgumentException();
    	 }
    	 return edges.get(e).tar;
     }
}
















